// Copyright (c) 2026 Huntr Software LLC
// SPDX-License-Identifier: MIT

#ifndef LTHEME_H
#define LTHEME_H

#include <filesystem>

#include <nlohmann/json.hpp>
using json = nlohmann::json;

#include "layers_global.h"
#include "layers_exports.h"

//#include "ljson.h"
#include "lstring.h"
#include "lstyle.h"

LAYERS_NAMESPACE_BEGIN

class LAttribute;

/*
	Represents a color theme in the Layers framework.

	A theme is a named collection of color attributes. Themes extend LStyle
	but typically contain only attributes (no child styles).

	> Most themes are defined in JSON files rather than created
	programmatically. For JSON syntax and the active theme mechanism,
	see the [Layers README](/Layers/). This documentation covers the C++
	interface for working with themes in code.

	### Creating Themes

	Themes are constructed with a name and publisher:
	```cpp
	LTheme* ocean = 
	    new LTheme("Ocean", "My Company");
	```

	New themes come with default attributes: `Foreground`, `Gradient`,
	`Primary`, `Secondary`, and `Tertiary`.

	### Accessing Colors

	Since themes extend LStyle, access attributes the same way:
	```cpp
	LAttribute* primary =
	    ocean->find_attribute("Primary");

	LString color = primary->as<LString>();
	// color = "#ffffff"
	```

	### Metadata

	Themes track publisher and UUID for identification:
	```cpp
	ocean->publisher();
	// "My Company"

	ocean->uuid();
	// "67fc1ff4-b6e8-4858-b7e5-0bcfb0227f2a"
	// ... or similar

	ocean->display_id();
	// "Ocean (My Company)"
	```
*/
class LAYERS_EXPORT LTheme : public LStyle
{
public:
	/*
		Constructs an empty theme.
	*/
	LTheme();

	/*
		Constructs a theme with the given name and publisher.
	*/
	LTheme(const LString& name, const LString& publisher);

	/*
		Constructs a theme from JSON data.

		The name parameter identifies the theme, value contains the JSON definition,
		and file_path indicates where the theme was loaded from.
	*/
	LTheme(
		const LString& name,
		const json& value,
		const std::filesystem::path& file_path);

	~LTheme();

	/*
		Appends a theme ID to this theme's inheritance lineage.

		The lineage tracks the history of themes this theme is based on.
	*/
	void append_to_lineage(const LString& theme_id);

	/*
		Returns the directory path where this theme is stored.
	*/
	std::filesystem::path directory() const;

	/*
		Returns a display-friendly identifier for this theme.
	*/
	LString display_id() const;

	/*
		Returns true if this theme can be edited by users.
	*/
	bool editable() const;

	/*
		Returns true if this theme has a custom implementation for the given application.
	*/
	bool has_implementation(const LString& app_display_id) const;

	/*
		Returns the lineage of theme IDs this theme is based on.
	*/
	std::vector<LString> lineage() const;

	/*
		Returns the publisher name for this theme.
	*/
	LString publisher() const;

	/*
		Saves this theme to disk.

		Only editable themes can be saved.
	*/
	void save();

	/*
		Saves the theme's metadata file to disk.
	*/
	void save_meta_file();

	/*
		Sets the directory where this theme should be stored.
	*/
	void set_dir(const std::filesystem::path& directory);

	/*
		Enables or disables edit mode for this theme.

		When enabled, changes to the theme will be saved.
	*/
	void set_edit_mode_enabled(bool enabled = true);

	/*
		Sets the publisher name for this theme.
	*/
	void set_publisher(const LString& publisher);

	/*
		Converts this theme to a JSON object representation.
	*/
	json to_json_object() const;

	/*
		Returns the unique identifier (UUID) for this theme.
	*/
	LString uuid() const;

private:
	class Impl;
	Impl* pimpl;
};
LAYERS_NAMESPACE_END

#endif // LTHEME_H
