// Copyright (c) 2026 Huntr Software LLC
// SPDX-License-Identifier: MIT

#ifndef LSTYLE_H
#define LSTYLE_H

#include <deque>
#include <set>

#include <nlohmann/json.hpp>
using json = nlohmann::json;

#include "layers_global.h"
#include "layers_exports.h"

#include "lstring.h"
#include "lobject.h"

LAYERS_NAMESPACE_BEGIN

class LAttribute;
class LStyle;
using LStyleList = std::vector<LStyle*>;
using LStyleMap = std::map<LString, LStyle*>;

/*
	Represents a UI element style in the Layers theme framework.

	Styles compose attributes and child styles to define the appearance of UI
	components.

	> Most styles are defined in JSON files rather than created
	programmatically. For JSON syntax and concepts,	see the
	[Layers README](/Layers/). This documentation covers the C++ interface for
	working with styles in code.

	Consider the following style:
	```cpp
	LStyle* btn = new LStyle("Button");
	```
	
	### Attributes

	Styles contain attributes that define visual properties.

	Create attributes using the `lMake<T>()` factory function to ensure the
	parent-child relationship is established properly:
	```cpp
	lMake<LAttribute>(btn, "Fill", "#ffffff");
	lMake<LAttribute>(btn, "Radius", 5.0);
	```

	Find attributes with `find_attribute()`:
	```cpp
	LAttribute* btn_fill =
	    btn->find_attribute("Fill");

	LString f = btn_fill->as<LString>();
	// f = "#ffffff"
	```

	You can iterate through all `attributes()`:
	```cpp
	cout << "Button Attributes:" << endl;
	for (auto& [name, attr] : btn->attributes())
	    cout << "  - " << name << endl;
	```

	### Children

	Styles can contain child styles representing sub-elements.

	Also create child styles using the `lMake<T>()` factory function:
	```cpp
	lMake<LStyle>(btn, "Icon Label");
	lMake<LStyle>(btn, "Text Label");
	```

	Find them with `find_item()`:
	```cpp
	LStyle* btn_icon =
	    btn->find_item("Icon Label");
	
	LStyle* btn_text =
	    btn->find_item("Text Label");
	```

	You can also iterate through all `children()`:
	```cpp
	cout << "Button Child Styles:" << endl;
	for (auto& [name, child] : btn->children())
	    cout << "  - " << name << endl;
	```

	### Paths

	Like attributes, every style has a path identifying its location in the
	hierarchy:
	```cpp
	btn->path();
	// "Button"

	btn_icon->path();
	// "Button/Icon Label"

	btn_icon->find_attribute("Svg Color")->path();
	// "Button/Icon Label/Svg Color"
	```

	### Inheritance

	Styles can `inherit()` from a base style and acquire its attributes and
	children:
	```cpp
	LStyle* exit_btn = new LStyle("Exit Button");
	lMake<LAttribute>(exit_btn, "Fill", "#ab5050");
	exit_btn->inherit(btn);
	```

	Since `Exit Button` has its own `Fill`, it doesn't inherit `Button/Fill`.

	Get a list of all `dependencies()`, which are all of the base styles
	inherited by the caller and its children:
	```cpp
	LStyle* lbl = new LStyle("Label");
	btn_icon->inherit(lbl);
	btn_text->inherit(lbl);

	auto deps = names(exit_btn->dependencies());
	// deps = { "Button", "Label" }
	```

	### Applying User Styles

	User styles are toggleable customizations that override styles:
	```cpp
	// Create user style that removes rounded corners:
	LStyle* no_radius = new LStyle("No Radius");
	lMake<LAttribute>(no_radius, "Radius", 0.0);

	// Then apply it:
	btn->apply_style(no_radius);
	```

	If you want to restore back to the original:
	```cpp
	btn->clear_style();
	```

	You can also register a callback to detect when styles are applied:
	```cpp
	btn->on_style_applied([]()
	    { cout << "Style changed!" << endl; });
	```

	### Extensions

	Extensions store arbitrary data with a style:
	```cpp
	if (btn->has_extension("tooltip"))
	{
	    json tooltip = btn->extension("tooltip");
	    // Use tooltip data
	}
	```

	Extensions can only be added in JSON since there is currently no public
	interface for adding them in code.
*/
class LAYERS_EXPORT LStyle : public LObject
{
public:
	/*
		Constructs an empty style.
	*/
	LStyle();

	/*
		Constructs a style with a name.
	*/
	LStyle(const LString& name);

	/*
		Constructs a style from JSON data.

		The name parameter identifies the style, value contains the JSON definition,
		and file_path indicates where the style was loaded from.
	*/
	LStyle(
		const LString& name,
		const json& value,
		const std::filesystem::path& file_path);

	~LStyle();

	/*
		Applies the properties of another style to this style.

		This is used for user styles and custom style overrides.
	*/
	void apply_style(LStyle* style_def);

	/*
		Returns the names of all attribute groups in this style.
	*/
	std::vector<LString> attribute_group_names() const;

	/*
		Returns a map of all attributes in this style.

		If type_index is specified, only returns attributes of that variant type.
	*/
	std::map<LString, LAttribute*> attributes(int type_index = -1) const;

	/*
		Returns the base style this style inherits from.

		Returns nullptr if this style has no base.
	*/
	LStyle* base() const;

	/*
		Returns the name of the base style.
	*/
	LString base_name() const;

	/*
		Returns the child style at the given index.

		Returns nullptr if the index is out of bounds.
	*/
	LStyle* child(int index) const;

	/*
		Returns the number of child styles.
	*/
	size_t child_count() const;

	/*
		Returns a map of all child styles.
	*/
	std::map<LString, LStyle*> children() const;

	/*
		Clears the applied custom style, restoring default values.
	*/
	void clear_style();
	
	void clone_to(LObject* parent);

	/*
		Returns a set of all styles this style depends on.
	*/
	std::set<LStyle*> dependencies();

	/*
		Returns the value of a custom extension property.

		Extensions allow storing arbitrary JSON data with a style.
	*/
	json extension(const LString& key) const;

	/*
		Returns all extension properties for this style.
	*/
	const json& extensions() const;

	/*
		Returns the filename this style was loaded from.
	*/
	LString file_name() const;

	/*
		Finalizes the style after loading and inheritance resolution.

		This locks the style structure and prepares it for use.
	*/
	void finalize();

	/*
		Finds an attribute by name within this style.

		Returns nullptr if the attribute is not found.
	*/
	LAttribute* find_attribute(const LString& attr_name);

	/*
		Finds a child style by path.

		The path uses '/' as a separator for nested styles.
	*/
	LStyle* find_item(const LString& path);

	/*
		Finds a child style by a list of names.
	*/
	LStyle* find_item(std::deque<LString> name_list);

	/*
		Returns true if this style has the specified extension property.
	*/
	bool has_extension(const LString& key) const;

	/*
		Returns true if this style has a base style that hasn't been resolved yet.
	*/
	bool has_unresolved_base() const;

	/*
		Returns this style's index among its siblings.
	*/
	int index() const;

	/*
		Returns true if this style can be overridden by user styles.
	*/
	bool is_overridable() const;

	/*
		Registers a callback to be invoked when a custom style is applied.
	*/
	void on_style_applied(std::function<void()> callback);

	/*
		Returns the full path of this style.
	*/
	LString path() const;

	/*
		Returns the parent style in the hierarchy.

		Returns nullptr if this is a root style.
	*/
	LStyle* parent() const;

	/*
		Returns the publisher name for this style.
	*/
	LString publisher() const;

	/*
		Resolves all attribute links within this style.
	*/
	void resolve_links();

	/*
		Sets the base style for inheritance.
	*/
	void inherit(LStyle* base_style);

	/*
		Sets the publisher name for this style.
	*/
	void set_publisher(const LString& publisher);

	/*
		Returns the custom style that has been applied to this style.

		Returns nullptr if no custom style is applied.
	*/
	LStyle* style() const;

	/*
		Converts this style to a JSON object representation.
	*/
	json to_json_object() const;

private:
	class Impl;
	Impl* pimpl;

	bool m_finalized{ false };
};

LAYERS_NAMESPACE_END

#endif // LSTYLE_H