// Copyright (c) 2026 Huntr Software LLC
// SPDX-License-Identifier: MIT

#ifndef LRESOURCES_H
#define LRESOURCES_H

#include "layers_global.h"
#include "layers_exports.h"

#include <map>

#include "lstring.h"

LAYERS_NAMESPACE_BEGIN

/*
	Represents a compiled resource (style file) embedded in the binary.

	Resources are generated by the Layers Resource Compiler and compiled
	into the application or library.
*/
struct LResource
{
    const unsigned char* data;
    size_t size;
};

/*
	Manages compiled resources generated by the Layers Resource Compiler.

	The resource manager provides access to styles and other resources that
	have been compiled into the binary through the .lrc build process.
	Use the lResourceManager macro to access the singleton instance.
*/
class LAYERS_EXPORT LResourceManager
{
public:
	/*
		Initializes the resource manager with compiled resource entries.

		This is called automatically by the generated resource initialization code.
	*/
    void init(const void* entries, size_t count);

	/*
		Returns the singleton resource manager instance.

		Prefer using the lResourceManager macro instead.
	*/
    static LResourceManager& instance();

	/*
		Retrieves a single resource by its path.

		Returns an LResource with null data if the resource is not found.
	*/
    LResource resource(const LString& path) const;

	/*
		Retrieves all resources under the given path prefix.

		Returns a map of resource paths to their data.
	*/
    std::map<LString, LResource> resources(const LString& path) const;

private:
    LResourceManager();
    ~LResourceManager();

    LResourceManager(const LResourceManager&) = delete;
    LResourceManager& operator=(const LResourceManager&) = delete;

    class Impl;
    Impl* pimpl;
};

/*
	Convenience macro for accessing the resource manager singleton.
*/
#define lResourceManager (Layers::LResourceManager::instance())

/*
	Convenience macro for accessing the resource() method.
*/
#define LRC (Layers::LResourceManager::resource)

LAYERS_NAMESPACE_END

#endif // LRESOURCES_H
