// Copyright (c) 2026 Huntr Software LLC
// SPDX-License-Identifier: MIT

#ifndef LLINK_H
#define LLINK_H

#include "layers_global.h"
#include "layers_exports.h"

#include <map>
#include <vector>

#include "lstring.h"

LAYERS_NAMESPACE_BEGIN

class LAttribute;
class LStyle;

class LLink;
using LLinkList = std::vector<LLink*>;

/*
	Represents a link between attributes in the Layers theme framework.

	Links allow attributes to reference other attributes, enabling theme
	inheritance and value sharing. Links can be absolute (referencing a full
	path) or relative (resolved within a style context).
*/
class LAYERS_EXPORT LLink
{
public:
	/*
		Constructs a link with an absolute or relative path.

		The path parameter specifies the target attribute. If relative_path
		is provided, the link is resolved relative to a style context.
	*/
	LLink(const LString& path, const LString& relative_path = "");

	/*
		Constructs a link directly to an attribute.
	*/
	LLink(LAttribute* attribute);

	/*
		Copy constructor.
	*/
	LLink(const LLink& l);

	~LLink();

	/*
		Returns the attribute this link points to.

		Returns nullptr if the link has not been resolved.
	*/
	LAttribute* attribute() const;

	/*
		Returns the absolute path of the link target.
	*/
	LString path() const;

	/*
		Returns the relative path component of this link.

		Returns an empty string for absolute links.
	*/
	LString relative_path() const;

	/*
		Resolves this link to point to the given attribute.

		Returns true if the link was successfully resolved.
	*/
	bool resolve(LAttribute* attr);

	/*
		Resolves a relative link within the given style context.

		Returns the resolved attribute, or nullptr if resolution fails.
	*/
	LAttribute* resolve(LStyle* context);

private:
	class Impl;
	Impl* pimpl;
};

LAYERS_NAMESPACE_END

#endif // LLINK_H
