// Copyright (c) 2026 Huntr Software LLC
// SPDX-License-Identifier: MIT

#ifndef LCONTROLLER_H
#define LCONTROLLER_H

#include <deque>
#include <map>
#include <filesystem>
#include <functional>
#include <vector>

#include "layers_global.h"
#include "layers_exports.h"

#include "lstring.h"

LAYERS_NAMESPACE_BEGIN

class LStyle;
class LTheme;

using LStyleList = std::vector<LStyle*>;

/*
	Central manager for themes and styles in the Layers framework.

	The controller loads, manages, and provides access to all themes and
	styles. It handles inheritance resolution, link resolution, and the
	active theme mechanism.

	> For related JSON syntax, file organization, and resource compiling,
	see the [Layers README](/Layers/). This documentation covers the C++
	interface for working with the controller in code.

	A default controller instance is available via the `lController` macro.

	### Loading Themes

	Load themes from a directory containing JSON theme files:
	```cpp
	lController.load_themes("/path/to/themes");
	```

	Access loaded themes by name:
	```cpp
	LTheme* dark = lController.theme("Dark");
	LTheme* light = lController.theme("Light");
	```

	Or iterate through all `themes()`:
	```cpp
	for (auto& [name, theme] : lController.themes())
	    cout << name << endl;
	```

	### The Active Theme

	The active theme is what `"Theme/..."` links resolve to.

	Set it with `set_active_theme()`:
	```cpp
	lController.set_active_theme(dark);
	```

	Access it with the `lTheme` macro:
	```cpp
	cout << lTheme->object_name() << endl;
	// "Dark"
	```

	When you change the active theme, all `"Theme/..."` links automatically
	update to reference the new theme's colors.

	### Loading Developer Styles

	Compiled developer styles should be loaded via library initializers to
	ensure proper load order:
	```cpp
	// mylib_init.cpp

	namespace Layers::Resources {
	    extern void initResources_styles();
	}

	class Initializer {
	public:
	    Initializer() {
	        Layers::Resources::initResources_styles();
	        lController.include("/styles/mylib");
	    }
	} static initializer;
	```

	This ensures styles load before `main()` and in dependency order.

	### Loading User Styles

	User styles are toggleable customizations loaded from a directory:
	```cpp
	lController.load_user_styles("/path/to/user/styles");
	```

	Toggle them on or off by name:
	```cpp
	lController.toggle_custom_style("Square Corners");
	```

	Get a list of currently active custom styles:
	```cpp
	LStyleList active = lController.active_custom_styles();
	```

	### Finding Styles

	Find styles by path:
	```cpp
	LStyle* button = lController.find_style("Button");

	LAttribute* fill = button->find_attribute("Fill");
	```

	Iterate through all loaded styles:
	```cpp
	for (auto& [path, style] : lController.styles())
	    cout << path << endl;
	```

	### Change Notifications

	Register a callback to detect when themes are added:
	```cpp
	lController.on_theme_added([](LTheme* theme)
	    { cout << "Added: " << theme->object_name() << endl; });
	```

	### Loading Order

	The proper sequence matters:

	1. Load developer styles (via library initializers)
	2. Load themes with `load_themes()`
	3. Set active theme with `set_active_theme()`
	4. Load user styles with `load_user_styles()` (optional)

	This ensures inheritance resolves and links establish correctly.
*/
class LAYERS_EXPORT LController
{
public:
    LController();
    ~LController();
    LController(const LController&) = delete;
    LController& operator=(const LController&) = delete;

	/*
		Returns a list of all currently active custom styles.
	*/
    LStyleList active_custom_styles();

	/*
		Returns the currently active theme.

		Returns nullptr if no theme is active. Prefer using the lTheme macro.
	*/
    LTheme* active_theme() const;

	/*
		Adds a theme to the controller.

		The controller takes ownership of the theme.
	*/
    void add_theme(std::unique_ptr<LTheme> theme);

	/*
		Finds a style by its path.

		Returns nullptr if the style is not found.
	*/
    LStyle* find_style(const LString& path);

	/*
		Finds a style by a list of path components.

		Returns nullptr if the style is not found.
	*/
	LStyle* find_style(std::deque<LString> name_list);

	/*
		Loads compiled styles from the given internal resource path.

		This is typically called from library initializers. If is_application
		is true, marks the styles as application-level.
	*/
    void include(const LString& path, bool is_application = false);

	/*
		Loads compiled internal resources at the given path.

		Used internally for loading framework resources.
	*/
    void include_internal(const LString& path);

	/*
		Returns the singleton controller instance.

		Prefer using the lController macro instead.
	*/
    static LController& instance();

	/*
		Loads a theme from a JSON string.

		Returns a unique pointer to the loaded theme.
	*/
    std::unique_ptr<LTheme> load_theme(const std::string& file_string);

	/*
		Loads all theme files from the given directory path.
	*/
	void load_themes(const std::filesystem::path& path);

	/*
		Loads user-created custom styles from the given directory path.

		User styles can override developer styles for customization.
	*/
    void load_user_styles(const std::filesystem::path& path);

	/*
		Registers a callback to be invoked when a new theme is added.

		The callback receives a pointer to the newly added theme.
	*/
	void on_theme_added(std::function<void(LTheme*)> callback);

	/*
		Returns the root style that contains all loaded styles.
	*/
    LStyle* root_style() const;

	/*
		Sets the active theme.

		Returns true if the theme was successfully activated. All attributes
		linking to "Theme/..." will now reference this theme.
	*/
    bool set_active_theme(LTheme* theme);

	/*
		Returns a map of all loaded styles.
	*/
    std::map<LString, std::unique_ptr<LStyle>>& styles() const;

	/*
		Finds a theme by its ID.

		Returns nullptr if the theme is not found.
	*/
    LTheme* theme(const LString& themeId) const;

	/*
		Returns a map of all loaded themes.
	*/
    std::map<LString, std::unique_ptr<LTheme>>& themes() const;

	/*
		Toggles a custom user style on or off.

		Returns true if the style was found and toggled successfully.
	*/
    bool toggle_custom_style(const LString& style_id);

private:
    class Impl;
    Impl* pimpl;
};

/*
	Convenience macro for accessing the controller singleton.
*/
#define lController (Layers::LController::instance())

/*
	Convenience macro for accessing the active theme.
*/
#define lTheme (Layers::LController::instance().active_theme())

LAYERS_NAMESPACE_END

#endif // LCONTROLLER_H
