// Copyright (c) 2026 Huntr Software LLC
// SPDX-License-Identifier: MIT

#ifndef LCONNECTOR_H
#define LCONNECTOR_H

#include "layers_global.h"
#include "layers_exports.h"

#include "lconnections.h"

LAYERS_NAMESPACE_BEGIN

/*
	Template class for managing callback connections with type-safe arguments.

	LConnector allows registering multiple callbacks that will be invoked
	when execute() is called. Each callback receives the arguments specified
	by the Args template parameters.
*/
template <typename... Args>
class LAYERS_EXPORT LConnector
{
public:
  using Callback = std::function<void(Args...)>;

private:
  template<typename... T>
  class Impl {
  public:
    LConnectionID next_connection_id{ 0 };
    std::map<LConnectionID, std::function<void(T...)>> connections;
  };

  Impl<Args...>* pimpl;

public:
  LConnector() : pimpl(new Impl<Args...>()) {}
  ~LConnector() { delete pimpl; }

  /*
	Registers a callback to be invoked when execute() is called.

	Returns a connection ID that can be used to disconnect the callback later.
  */
  LConnectionID connect(Callback callback)
  {
    pimpl->connections[pimpl->next_connection_id] = callback;
    return pimpl->next_connection_id++;
  }

  /*
	Disconnects and removes the callback associated with the given connection ID.
  */
  void disconnect(const LConnectionID& connection)
  {
    pimpl->connections.erase(connection);
  }

  /*
	Disconnects and removes all registered callbacks.
  */
  void disconnect_all()
  {
    pimpl->connections.clear();
  }

  /*
	Executes all registered callbacks with the provided arguments.

	Callbacks are invoked in the order they were connected.
  */
  void execute(Args... args)
  {
    for (auto& [id, callback] : pimpl->connections)
    {
      callback(args...);
    }
  }
};

LAYERS_NAMESPACE_END

#endif // LCONNECTOR_H
