// Copyright (c) 2026 Huntr Software LLC
// SPDX-License-Identifier: MIT

#ifndef LCONNECTIONS_H
#define LCONNECTIONS_H

#include <functional>
#include <map>

#include "layers_global.h"
#include "layers_exports.h"

LAYERS_NAMESPACE_BEGIN

/*
	Unique identifier for callback connections.

	Used to track and manage callback connections so they can be
	disconnected later when no longer needed.
*/
class LAYERS_EXPORT LConnectionID
{
public:
	/*
		Constructs an invalid connection ID.
	*/
	LConnectionID();

	/*
		Constructs a connection ID with the given value.
	*/
	LConnectionID(int value);

	LConnectionID(const LConnectionID& other) = default;

	/*
		Assigns the value of another connection ID to this one.
	*/
	LConnectionID& operator=(const LConnectionID& other);

	/*
		Compares this connection ID with another for ordering.
	*/
	bool operator<(const LConnectionID& other) const;

	/*
		Checks if this connection ID equals another.
	*/
	bool operator==(const LConnectionID& other) const;

	/*
		Post-increment operator. Returns the current ID and increments it.
	*/
	LConnectionID operator++(int);

private:
	int m_value;
};

/*
	Map of connection IDs to their callback functions.

	Used internally to manage multiple callback connections.
*/
using LConnections = std::map<LConnectionID, std::function<void()>>;

LAYERS_NAMESPACE_END

#endif // LCONNECTIONS_H
