// Copyright (c) 2026 Huntr Software LLC
// SPDX-License-Identifier: MIT

#ifndef LATTRIBUTE_H
#define LATTRIBUTE_H

#include <functional>
#include <map>
#include <string>
#include <variant>
#include <vector>

#include <nlohmann/json.hpp>
using json = nlohmann::json;

#include "layers_global.h"
#include "layers_exports.h"

#include "lconnections.h"
#include "lcontroller.h"
#include "llink.h"
#include "lobject.h"
#include "lstring.h"

LAYERS_NAMESPACE_BEGIN

/*
	Variant type that can hold attribute values.
	Can be: monostate (empty), double, bool, LString, or vector of LStrings.
*/
using LVariant = std::variant<
	std::monostate,				// 0
	double,						// 1
	bool,						// 2
	LString,					// 3
	std::vector<LString>>;		// 4

class LAttribute;
using LAttributeList = std::vector<LAttribute*>;
using LAttributeMap = std::map<LString, LAttribute*>;

/*
	Represents a named value within the Layers theme framework.

	Attributes are the building blocks of themes and styles. They store
	numbers, strings, booleans, or arrays through the LVariant type.

	> Most attributes are defined in JSON themes and styles rather than created
	programmatically. For JSON syntax and concepts, see the 
	[Layers README](/Layers/). This documentation covers the C++ interface for
	working with attributes in code.

	### Values

	Create attributes with different value types:
	```cpp
	LAttribute* fill =
	    new LAttribute("Fill", "#ffffff");
	
	LAttribute* thickness =
	    new LAttribute("Thickness", 5.0);
	```

	Access values with `as<T>()`:
	```cpp	
	LString f = fill->as<LString>();
	// f = "#ffffff"

	double t = thickness->as<double>();
	// t = 5.0
	```

	For safe access, `as_if<T>()` returns *nullptr* if the type doesn't match:
	```cpp
	if (const auto* f = fill->as_if<LString>())
	    // fill must be an LString
	```

	### Paths

	Since attributes are owned, they exist within an *object hierarchy*.

	Every attribute has a path identifying its location in the hierarchy.

	Paths use `/` to separate between object names:
	```
	// Assume fill belongs to a button
	"Button/Fill"
	```

	### Linking

	Linking lets one attribute reference another, further known as the *link
	attribute*. When linked, valuation returns the link attribute's
	value:
	```cpp	
	LAttribute* primary =
	    new LAttribute("Primary", "#001122");
	
	fill->create_link(primary);

	f = fill->as<LString>();
	// f = #001122 (value indirectly from primary)
	```

	An attribute that links to another is known as a *dependent attribute*.

	You can acquire a list of dependent attributes:
	```cpp
	LAttributeList deps = primary->dependent_attributes();
	// deps contains the fill attribute
	```

	Remember to break links when they are no longer needed. When a link is
	broken, the attribute copies its value:
	```cpp
	fill->break_link();

	f = fill->as<LString>();
	// f = #001122 (same value directly from fill now)

	LAttributeList deps = primary->dependent_attributes();
	// deps no longer contains the fill attribute
	```

	### States

	States let attributes have multiple values, accessed conditionally.

	States are just sub-attributes in the object hierarchy:
	```cpp
	lMake<LAttribute>(fill, "Selected", "#445566");
	// fill is a parent receiving a "Selected" state
	```

	You can access a state value directly:
	```cpp
	f = fill->state("Hovered")->as<LString>();
	// f = #445566 (state value accessed directly)
	```

	However, prefer passing a state variable when accessing the parent's value:
	```cpp
	// State variable, subject to change
	LString state = "Selected";

	f = fill->as<LString>(state);
	// f = #445566 (state value accessed through fill)
	```

	You can check if an attribute has any states:
	```cpp
	if (fill->has_states())
	    // fill must have states
	```

	### Change Notifications

	Register callbacks to detect value changes:
	```cpp
	LConnectionID conn = fill->on_change([]()
	    { cout << "Fill changed!" << endl; });

	// Remember to disconnect when it's no longer needed
	fill->disconnect_change(conn);
	```
*/
class LAYERS_EXPORT LAttribute : public LObject
{
public:
	/*
		Constructs an attribute with the given name.
	*/
	LAttribute(const LString& name);

	/*
		Constructs an attribute with the given name and numeric value.
	*/
	LAttribute(const LString& name, double value);

	/*
		Constructs an attribute with the given name and string value.
	*/
	LAttribute(const LString& name, const char* value);

	/*
		Constructs an attribute with the given name and variant value.
	*/
	LAttribute(const LString& name, const LVariant& value);

	/*
		Constructs an attribute with the given name and JSON value.
	*/
	LAttribute(const LString& name, const json& value);

	virtual ~LAttribute();

	/*
		Returns the attribute's value as the specified type T.

		If the attribute has states, state_combo is used to select the appropriate
		state. If the attribute is linked, the value is retrieved from the
		link attribute.
		
		The context parameter is used for resolving relative links.
	*/
	template<typename T>
	T as(
		const LStringList& state_combo = LStringList(),
		LStyle* context = nullptr);

	/*
		Returns a pointer to the attribute's value if it matches type T,
		otherwise returns nullptr.

		Provides safe type checking before accessing attribute values.
	*/
	template<typename T>
	const T* as_if(
		const LStringList& state_combo = LStringList(),
		LStyle* context = nullptr);

	/*
		Breaks the link connection to another attribute.

		If update is true, notifies dependent attributes of the change.
	*/
	void break_link(bool update = true);

	/*
		Clears the style attribute association.
	*/
	void clear_style_attribute();

	void clone_to(LObject* parent);

	/*
		Creates a link to another attribute.

		When linked, this attribute's value is derived from link_attr.
	*/
	void create_link(LAttribute* link_attr);

	/*
		Creates a link using an LLink object.
	*/
	void create_link(LLink* link);

	/*
		Returns a list of attributes that depend on this attribute.

		If include_indirect_dependencies is true, includes transitive dependencies.
	*/
	LAttributeList dependent_attributes(
		bool include_indirect_dependencies = false) const;

	/*
		Disconnects the change callback associated with connection.
	*/
	void disconnect_change(const LConnectionID& connection);

	/*
		Disconnects the link change callback associated with connection.
	*/
	void disconnect_link_change(const LConnectionID& connection);

	/*
		Returns true if this attribute has state variants.
	*/
	bool has_states() const;

	/*
		Registers a callback to be invoked when the attribute value changes.

		Returns a connection ID that can be used to disconnect the callback.
	*/
	LConnectionID on_change(std::function<void()> callback);

	/*
		Registers a callback to be invoked when the attribute's link changes.

		Returns a connection ID that can be used to disconnect the callback.
	*/
	LConnectionID on_link_change(std::function<void()> callback);

	/*
		Returns the full path of this attribute.
	*/
	LString path() const;

	/*
		Resolves all link references within this attribute.
	*/
	void resolve_links();

	/*
		Associates this attribute with a style attribute.

		Style attributes provide default values that can be overridden.
	*/
	void set_style_attribute(LAttribute* style_attribute);

	/*
		Sets the attribute value from a C-string.
	*/
	void set_value(const char* value);

	/*
		Sets the attribute value from a variant.
	*/
	void set_value(const LVariant& value);

	/*
		Returns the state attribute matching the given state combination.

		Returns nullptr if no matching state exists.
	*/
	LAttribute* state(const LStringList& state_combo);

	/*
		Returns a map of all state attributes.

		If include_parent_states is true, includes inherited states.
	*/
	LAttributeMap states(bool include_parent_states = true) const;

	/*
		Returns the link object if this attribute links to another attribute.

		Returns nullptr if this attribute is not a link.
	*/
	LLink* link() const;

	/*
		Returns the style attribute associated with this attribute.

		Returns nullptr if no style attribute is set.
	*/
	LAttribute* style_attribute() const;

	/*
		Converts this attribute to a JSON object representation.
	*/
	json to_json_object() const;

	/*
		Converts this attribute's value to a JSON value.
	*/
	json to_json_value() const;

	/*
		Returns the type index of the current value in the variant.
	*/
	size_t type_index() const;

	/*
		Returns a reference to the underlying variant value.
	*/
	const LVariant& value();

private:
	void update_link_dependencies();

	class Impl;
	Impl* pimpl;
};

template<typename T>
T LAttribute::as(const LStringList& state_combo, LStyle* context)
{
	if (style_attribute())
		return style_attribute()->as<T>(state_combo);
	
	if (!states().empty() && !state_combo.empty())
		if (LAttribute* state_attr = state(state_combo))
			return state_attr->as<T>();
	
	if (link())
	{
		if (link()->attribute())
		{
			return link()->attribute()->as<T>(state_combo, context);
		}
		else if (link()->relative_path() != "")
		{
			if (LAttribute* res_attr = link()->resolve(context))
			{
				return res_attr->as<T>();
			}
		}
	}

	return std::get<T>(value());
}

template<typename T>
const T* LAttribute::as_if(const LStringList& state_combo, LStyle* context)
{
	if (style_attribute())
		return style_attribute()->as_if<T>(state_combo);

	if (!states().empty() && !state_combo.empty())
		if (LAttribute* state_attr = state(state_combo))
			return state_attr->as_if<T>();

	if (link())
	{
		if (link()->attribute())
		{
			return link()->attribute()->as_if<T>(state_combo, context);
		}
		else if (link()->relative_path() != "")
		{
			if (LAttribute* res_attr = link()->resolve(context))
			{
				return res_attr->as_if<T>();
			}
		}
	}

	return std::get_if<T>(&value());
}

LAYERS_NAMESPACE_END

#endif // LATTRIBUTE_H
