// Copyright (c) 2026 Huntr Software LLC
// SPDX-License-Identifier: MIT

#ifndef LALGORITHMS_H
#define LALGORITHMS_H

#include <filesystem>
#include <iostream>
#include <fstream>
#include <sstream>
#include <vector>
#include <string>

#include "layers_global.h"
#include "layers_exports.h"

#include "lstring.h"

LAYERS_NAMESPACE_BEGIN

/*
	Splits a string into tokens based on a delimiter.

	The template parameter T should be a container type (e.g., std::vector<LString>).
	Returns a container of LString tokens.
*/
template<typename T>
inline LAYERS_EXPORT T split(const LString& s, char delimiter)
{
	T tokens;
	std::string token;
	std::istringstream tokenStream(s);
	while (std::getline(tokenStream, token, delimiter)) {
		tokens.push_back(LString(token));
	}
	return tokens;
}

/*
	Removes whitespace (spaces, newlines, carriage returns) from a string,
	preserving whitespace within quoted sections.

	Returns a new string with whitespace removed.
*/
inline LAYERS_EXPORT
std::string remove_whitespace(const std::string& str)
{
	std::string result;

	bool insideQuotes = false;

	for (char c : str)
	{
		if (c == '\"')
			insideQuotes = !insideQuotes;

		if (insideQuotes || (c != ' ' && c != '\n' && c != '\r'))
			result.push_back(c);
	}

	return result;
}

/*
	Loads a JSON file from disk and returns its contents with whitespace removed.

	Returns the file contents as a string, or an empty string if the file
	cannot be opened.
*/
inline LAYERS_EXPORT
std::string load_json_file(const std::filesystem::path& path)
{
	std::ifstream file(path, std::ios::in);
	if (!file.is_open())
	{
		std::cerr << __FUNCTION__ << ": Could not read file: "
			<< path.string() << std::endl;
		return "";
	}

	std::stringstream buffer;
	buffer << file.rdbuf();
	std::string data = buffer.str();
	file.close();

	data = remove_whitespace(data);

	return data;
}

LAYERS_NAMESPACE_END

#endif // LALGORITHMS_H
