# ![Layers Logo Image](https://codeberg.org/HuntrSoftware/Layers/raw/branch/main/images/layers_logo.svg) Layers

> ⚠️ **This project is still under active development. It is not yet recommended for production use. Developers should proceed with caution. Please report any issues on [Codeberg](https://codeberg.org/HuntrSoftware).**

Layers is a C++ theme framework that separates UI styling from application logic. It introduces _**attributes**_, _**themes**_, and _**styles**_ stored in [JSON format](https://en.wikipedia.org/wiki/JSON).

### Attributes

An attribute is a named value:

```
"Primary": "#333c4f"
```

Values can be complex, like this array of strings representing a gradient:

```
"Gradient": [
  "0:#363f57",
  "1:#424e66"
]
```

Attributes are identifiable via their paths.

A path is composed of the attribute's name prefixed with its parents' names separated by '**/**' characters:

```
"Theme/Primary"
```

Through linking, one value can represent many attributes.

A link is made by prefixing the target attribute's path with "**L:**".

Links replace values:

```
"Fill": "L:Theme/Primary"
```

### Themes

A theme is a named collection of color attributes:

```
{
  "Dark": {
    "Foreground": "#e3e3e3",
    "Gradient": [
      "0:#3a3c42",
      "1:#42454d"
    ],
    "Primary": "#36393f",
    "Secondary": "#2f3136",
    "Tertiary": "#25272b"
  }
}
```

These attributes can be linked to directly if the theme is certain to be loaded:

```
"Text Color": "L:Dark/Foreground"
```

Instead, the active theme can be linked to by using the reserved name "Theme":

```
"Text Color": "L:Theme/Foreground"
```

### Styles

A style represents a UI element by composing attributes and a hierarchy of child styles:

```
{
  "Button": {
    "Corner Radii": 3,
    "Fill": "L:Theme/Tertiary"
    "Icon Label/Svg/Color": "L:Theme/Foreground",
    "Text Label/Text Color": "L:Theme/Foreground"
  }
}
```

Styles are reusable through inheritance.

Inheritance is established by prefixing the base style's path with "**S:**". This can be passed either as a value directly or within an object with **"->"** as the key.

With inheritance, the above can be rearranged:

```
{
  "Control": {
    "Corner Radii": 3,
    "Fill": "L:Theme/Tertiary"
  },
  "Label": {
    "Svg/Color": "L:Theme/Foreground",
    "Text Color": "L:Theme/Foreground"
  },
  "Button": {
    "->": "S:Control",
    "(Icon,Text) Label": "S:Label"
  }
}
```

Styles can be divided into an arbitrary count of files.

This significantly improves readability by reducing the amount of content in each file:

```
// Button in its own button.json file
{
  "Button": {
    "->": "S:Control",
    "(Icon,Text) Label": "S:Label"
  }
}
```

#### Developer Styles

Developers are responsible for creating styles for their UI elements.

These styles should be stored along with project source-code:

```
MyApp/
├── CMakeLists.txt
├── src/
│   └── main.cpp
└── styles/               <-- Maintain styles here
    ├── .lrc              <-- Resource compiler config
    ├── controls.json
    ├── dialog.json
    └── logindialog.json
```

Layers includes a resource compiler because developer styles need to be compiled with their libraries/applications. 

The ```.lrc``` file lists the file-names of the styles to be compiled:

```
{
  "/styles/myapp": [
    "controls.json",
    "dialog.json",
    "logindialog.json"
  ]
}
```

Then, within in the build process, invoke the resource compiler and pass it the *.lrc* file:

```
LayersResourceCompiler LRC_FILE NEW_CPP_FILE
```

Projects building with CMake can use the following to generate a ```styles.cpp``` file and add it to the build target:

```
set(LRC "${LAYERS_DIR}/bin/LayersResourceCompiler")
set(LRC_D "${LAYERS_DIR}/bin/LayersResourceCompilerd")

set(LRC_CONF "${CMAKE_CURRENT_SOURCE_DIR}/styles/.lrc")
set(STYLES_CPP "${CMAKE_CURRENT_BINARY_DIR}/styles.cpp")

add_custom_command(
  OUTPUT ${STYLES_CPP}
  COMMAND $<$<CONFIG:Debug>:${LRC_D}>$<$<CONFIG:Release>:${LRC}>
    ${LRC_CONF}
    ${STYLES_CPP}
  DEPENDS ${LRC_CONF}
  COMMENT "Generating ${STYLES_CPP} from ${LRC_CONF}"
  VERBATIM
)

# If target is app:
add_executable(myapp
  ${GUI_SOURCES}
  ${STYLES_CPP}
)

# If target is lib:
add_library(mylib SHARED
  ${LIB_HEADERS}
  ${LIB_SOURCES}
  ${STYLES_CPP}
)
```

This generates the ```styles.cpp``` file and compiles it. The file is placed in the ```build/``` directory:

```
# After running CMake/Build:
MyApp/
├── build/
│   └── styles.cpp    <-- Generated C++ file (compiled styles)
└── styles/
    └── ...
```

Next, developers need to ensure that their styles get loaded. This can be handled with a **library-initializer**:

**mylib_init.h**:

```
namespace MyLib {
    class Initializer {
    public:
        Initializer();
    };
}
```
**mylib_init.cpp**:

```
#include <MyLib/mylib_init.h>
#include <Layers/lcontroller.h>

using namespace MyLib;

namespace Layers {
    namespace Resources {
        extern void initResources_styles();
    }
}

Initializer::Initializer()
{
    Layers::Resources::initResources_styles();
    lController.include_internal("/styles/mylib");
} static Initializer i;
```

The ```extern``` declaration references the function generated by the resource compiler.

The static instance ensures initialization happens before main().

#### User Styles

**Users styles** modify the developer ones. These can be created and toggled on by users to make fine-tuned changes to the UI.

Here is a user style that removes rounded corners from all styles:

```
{
  "Square Corners": {
    "**/Corner Radii": 0
  }
}
```

And here is a user style the gives rainbow borders to all top-level styles:

```
{
  "Rainbow Borders": {
    "*/Border.Fill": [
      "0:#ff4c4c",
      "0.167:#ffa24c",
      "0.333:#fff34c",
      "0.5:#82ff4c",
      "0.667:#4cfff3",
      "0.833:#4c6aff",
      "1:#f34cff"
    ]
  }
}
```

### The Layers Controller

The controller is responsible for loading and managing themes and styles.

The default controller is statically initialized, available via ```LController::instance()``` or the ```lController``` macro. This was demonstrated previously when loading compiled styles from the library-initializer:

```
#include <Layers/lcontroller.h>

// ... within MyLib::Initializer::Initializer():

lController.load_styles("/styles/mylib");
```

#### Loading styles

For compiled developer styles, it is highly recommended to **use the library-intializer method**. This method ensures that style inheritance resolution order is maintained **since libraries already load in order, naturally**.

User styles do not need to be compiled and can be loaded anytime during or after app initialization via ```lController.load_user_styles()```:

```
lController.load_user_styles("C:\Layers\Styles\");
```

##### Toggling user styles

Once user styles have been loaded, they can be toggled on & off:

```
LAttribute* attr = lController.find("Main Window")->
    find("Corner Radii.Top Left");

// Before:
cout << attr->path() << ": " << attr->as<LString>() << endl;
// Main Window/Corner Radii.Top Left: 8

// Remove all rounded corners
lController.toggle_user_style("Square Corners");

// After:
cout << attr->path() << ": " << attr->as<LString>() << endl;
// Main Window/Corner Radii.Top Left: 0
```

#### Resolving style inheritance

The controller resolves style inheritance. Target styles need to be available or resolution will fail.

If the target is from the same set of styles being loaded, then resolution should work fine. If the target is from another library, then those styles must be loaded *first*.

Note that if developers **use the library-initializer method** to load their compiled styles, then resolution should work fine **since dependencies naturally load in order**.

#### Loading themes

Themes can be loaded via ```lController.load_themes()```:

```
lController.load_themes("C:\Layers\Themes\");
```

##### Tracking the active theme

Once themes have been loaded, the *active theme* can be set with ```lController.set_active_theme()```, and its accessible via ```lController.active_theme()``` or the ```lTheme``` macro:

```
using namespace std;

lController.set_active_theme(
  lController.find("Blue"));

cout << "Active theme: " << lTheme.name() << endl;

// Active theme: Blue
```